import os
import json
import openai
import sqlite3


class Database:
    def __init__(self, config):
        self.config = config
        db_path = self.config.db_path
        self.db_name = os.path.basename(db_path).split('.')[0]
        if not os.path.exists(os.path.dirname(db_path)):
            os.makedirs(os.path.dirname(db_path))
        self.conn = sqlite3.connect(db_path)
        self.c = self.conn.cursor()
        self.create_table()

    def create_table(self):
        cmds = []
        for column in self.config.columns:
            cmds.append(f'{column} TEXT NOT NULL')
        cmds = ',\n'.join(cmds)
        self.c.execute(f'''
            CREATE TABLE IF NOT EXISTS {self.db_name} (
                id INTEGER PRIMARY KEY,
                {cmds}
            )
        ''')
        self.conn.commit()

    def add_row(self, *args, **kwargs):
        columns = ', '.join(self.config.columns)
        self.c.execute(f'''
            INSERT INTO {self.db_name} ({columns})
            VALUES (?, ?)
        ''', (*args,))
        self.conn.commit()

    def get_object_by_id(self, column_name, obj_id):
        self.c.execute(f'''
            SELECT {column_name} FROM {self.db_name} WHERE id=?
        ''', (obj_id,))
        return self.c.fetchone()[0]

    def update_row(self, row_id, *args, **kwargs):
        cmds = []
        for column in self.config.columns:
            cmds.append(f'{column} = ?')
        cmds = ', '.join(cmds)
        self.c.execute(f'''
            UPDATE {self.db_name}
            SET {cmds}
            WHERE id = ?
        ''', (*args, row_id))
        self.conn.commit()

    def delete_row(self, row_id):
        self.c.execute(f'''
            DELETE FROM {self.db_name} WHERE id=?
        ''', (row_id,))
        self.conn.commit()

    def list_rows(self):
        cmds = ', '.join(self.config.columns)
        self.c.execute(f'SELECT id, {cmds} FROM {self.db_name}')
        return self.c.fetchall()
    
    def list_columns(self, column_name):
        self.c.execute(f'SELECT id, {column_name} FROM {self.db_name}')
        return self.c.fetchall()

    def close(self):
        self.conn.close()

# Example usage
if __name__ == "__main__":
    openai.api_key = 'your-openai-api-key'
    
    def generate_embedding(chunk):
        response = openai.Embedding.create(
            input=chunk,
            model="text-embedding-ada-002"
        )
        return response['data'][0]['embedding']
    
    # Initialize the database
    db = Database()

    # Add a chunk
    chunk = "This is a sample chunk of text."
    embedding = generate_embedding(chunk)
    db.add_chunk(chunk, embedding)

    # List all chunks
    chunks = db.list_rows()
    print("Chunks in database:")
    for id, chunk, embedding in chunks:
        print(f"ID: {id}, Chunk: {chunk}, Embedding: {embedding}")

    # Update a chunk
    new_chunk = "This is an updated chunk of text."
    new_embedding = generate_embedding(new_chunk)
    db.update_chunk(1, new_chunk, new_embedding)

    # Get a chunk
    print("Retrieved chunk:")
    print(db.get_chunk(1))

    # Delete a chunk
    db.delete_row(1)

    # List all chunks after deletion
    print("Chunks in database after deletion:")
    chunks = db.list_rows()
    for id, chunk, embedding in chunks:
        print(f"ID: {id}, Chunk: {chunk}, Embedding: {embedding}")

    # Close the database
    db.close()
